Test_EdgesAndChains = class()

function Test_EdgesAndChains:init()
    self.title = "edges & chains"
end

function Test_EdgesAndChains:setup()
    self.lab = PhysicsLab()  
    
    -- define units that change with orientation
    local rltvX, rltvY = WIDTH/2, HEIGHT/2
    -- define a unit that is constant regardless of orientation
    -- calibrated to be 10.0 in portrait mode on iPad Pro
    local abslUnit = math.min(WIDTH, HEIGHT) * 0.009765625
    
    self:makeBox(abslUnit)
    
    local cartX, cartY = rltvX + abslUnit * 34, rltvY + abslUnit * 35
    self:makeCart(abslUnit, cartX, cartY)
        
    -- make slopes
    local leftSlopeStart = vec2(rltvX - abslUnit * 47.08352, rltvY - abslUnit * 0.856064)
    local leftSlopeEnd = vec2(rltvX + abslUnit * 8.56064, rltvY - abslUnit * 12.84096) 
    local numWavesL, waveHeightL = 0.49651712, abslUnit * 3.6168704
    self.lab:CHAIN_SineWaveBetweenPoints(leftSlopeStart, leftSlopeEnd, numWavesL, waveHeightL)
    
    local rightSlopeStart = vec2(rltvX + abslUnit * 55.64416, rltvY + abslUnit * 42.8032)
    local rightSlopeEnd = vec2(rltvX - abslUnit * 9.416704, rltvY + abslUnit * 5.992448)
    local numWavesR, waveHeightR = 0.49651712, abslUnit * 6.3990784
    self.lab:CHAIN_SineWaveBetweenPoints(rightSlopeStart, rightSlopeEnd, numWavesR, waveHeightR)
    
    local chunkyRoadStart = vec2(rltvX + abslUnit * 55.64416, rltvY - abslUnit * 8.56064)
    local chunkyRoadEnd = vec2(rltvX - abslUnit * 55.64416, rltvY - abslUnit * 55.64416)
    local numWavesCh, waveHeightCh, segmentLength = 1.9, abslUnit * -3.7725336, 80
    self.lab:CHAIN_SineWaveBetweenPoints(chunkyRoadStart, chunkyRoadEnd, numWavesCh, waveHeightCh, segmentLength)

    self:silenceCollisions() 
    
    print("EDGEs are lines that bodies collide with. A CHAIN is a linked series of edges.\n\nPhysicsLab lets you make smooth or blocky CHAINs in the shape of a repeating wave (or any part of one). All the roads in this test are made with those waves.")
end

function Test_EdgesAndChains:makeBox(unit)
    local sideSize = unit * 85.6064   
    local sideHalf = sideSize / 2   
    local boxLL, boxLR = vec2((WIDTH/2) - sideHalf, (HEIGHT/2) - sideHalf - 10), vec2((WIDTH/2) + sideHalf, (HEIGHT/2) - sideHalf - 10)
    local boxUL, boxUR = vec2((WIDTH/2) - sideHalf, (HEIGHT/2) + sideHalf - 10), vec2((WIDTH/2) + sideHalf, (HEIGHT/2) + sideHalf - 10)
    self.lab:collidableWall(boxLL.x - 18, boxLL.y, boxLR.x + 5, boxLR.y)
    self.lab:collidableWall(boxUL.x - 8, boxUL.y, boxUR.x + 11, boxUR.y)
    self.lab:collidableWall(boxUL.x , boxUL.y + 15, boxLL.x, boxLL.y - 9)
    self.lab:collidableWall(boxUR.x, boxUR.y + 8, boxLR.x, boxLR.y - 12)  
end

function Test_EdgesAndChains:makeCart(unit, x, y)
    local cartWidth = unit * 9.4
    local cartHeight = cartWidth * 0.6
    local wheelRadius = cartHeight * 0.3
    local wheelOffsetX = cartWidth * 0.3
    local wheelOffsetY = cartHeight * 0.5
    local cart = self.lab:boxAt(x, y, cartWidth, cartHeight)
    local leftWheel = self.lab:circleAt(cart.x - wheelOffsetX, cart.y - wheelOffsetY, wheelRadius)
    local rightWheel = self.lab:circleAt(cart.x + wheelOffsetX, cart.y - wheelOffsetY, wheelRadius)
    self.leftJ = self.lab:pointThatTwoBodiesRevolveAround(leftWheel.x, leftWheel.y, cart, leftWheel)
    self.rightJ = self.lab:pointThatTwoBodiesRevolveAround(rightWheel.x, rightWheel.y, cart, rightWheel)
end

function Test_EdgesAndChains:silenceCollisions()
    self.lab.collideFunction = function(contact)
        if contact.state == BEGAN then
            self.lab.contacts[contact.id] = contact
        elseif contact.state == MOVING then
            self.lab.contacts[contact.id] = contact
        elseif contact.state == ENDED then
            self.lab.contacts[contact.id] = nil
        end
    end
end

function Test_EdgesAndChains:draw()
    self.lab:draw()
end

function Test_EdgesAndChains:touched(touch)
    self.lab:touched(touch)
end

function Test_EdgesAndChains:collide(contact)
    self.lab:collide(contact)
end

function Test_EdgesAndChains:cleanup()
    self.lab:clear()
    self.leftJ = nil
end